/* By Bjorn Lindeijer and Laurens van der Starre */
#include <stdlib.h>
#include <stdio.h>
#include <malloc.h>
#include <math.h>
#include <GL/glut.h>

/* external gaussian random number generator, with zero mean and sigma = 1 */
extern float gasdev(int *idum);

typedef GLfloat Vertex3D[3];
typedef GLfloat Normal3D[3];

int winwidth = 600,
winheight = 512;

/* Iterations */

#define STEPS (9)
#define NSIDE ((1 << STEPS) + 1)

GLfloat ground[NSIDE][NSIDE];
GLfloat minHeight, maxHeight;
GLfloat waterHeight;


/* view point variables */ 

GLfloat alt = 45,                   /* angles */ 
azim = 45,
height = 0.25, distance = 2; /* location */


int idum = 42;
float speed = 0.0;
float alt_speed = 0.0;
float azim_speed = 1.0;
float fase = 0.0;
int detail = 2;                     /* drawing detail level */


/*
*  FUNCTIONS USED FOR DRAWING BELOW HERE
*/

void setColor(GLfloat height)
{
	static GLfloat green[3] = {0.1, 0.7, 0.2};
	static GLfloat brown[3] = {0.5, 0.3, 0.2};
	static GLfloat white[3] = {1.0, 1.0, 1.0};
	GLfloat whitei = maxHeight;
	GLfloat greeni = minHeight;
	GLfloat browni = (2*maxHeight + minHeight) / 3.0;
	GLfloat color[3];

	if (height > browni) {
		GLfloat factor = (height - browni) / (whitei - browni);
		color[0] = (1 - factor) * brown[0] + factor * white[0];
		color[1] = (1 - factor) * brown[1] + factor * white[1];
		color[2] = (1 - factor) * brown[2] + factor * white[2];
	}
	else {
		GLfloat factor = (height - greeni) / (browni - greeni);
		color[0] = (1 - factor) * green[0] + factor * brown[0];
		color[1] = (1 - factor) * green[1] + factor * brown[1];
		color[2] = (1 - factor) * green[2] + factor * brown[2];
	}

	//printf("Setting color (%f, %f, %f)\n", color[0], color[1], color[2]);
	glColor3f(color[0], color[1], color[2]);
}

void setNormal(int x, int y, int p)
{
	GLfloat dx, dy;

	if (x >= p && x < NSIDE - p - 1)
		dx = ground[x-p][y] - ground[x+p][y];
	else if (x < NSIDE - p - 1)
		dx = 2 * (ground[x][y] - ground[x+p][y]);
	else if (x >= p)
		dx = 2 * (ground[x-p][y] - ground[x][y]);

	if (y >= p && y < NSIDE - p - 1)
		dy = ground[x][y-p] - ground[x][y+p];
	else if (y < NSIDE - p - 1)
		dy = 2 * (ground[x][y] - ground[x][y+p]);
	else if (y >= p)
		dy = 2 * (ground[x][y-p] - ground[x][y]);

	glNormal3f(dx, dy, 1.0);
}

GLfloat getHeight(int x, int y)
{
	GLfloat height = ground[x][y];
	/*
	// Uncommenting this will enable an experimental wavy effect on the landscape
	float xf = (float)(x - 0.5 * NSIDE) / NSIDE;
	float yf = (float)(y - 0.5 * NSIDE) / NSIDE;
	float dist = sqrt(xf*xf + yf*yf);
	float factor = 4 * (0.5 - dist);
	if (factor < 0) factor = 0;
	height += sin(dist * 8 * 3.14 + fase) * factor;
	*/
	return height;
}

void placeGroundVertex(int x, int y, int p)
{
	setNormal(x, y, p);
	setColor(ground[x][y]);
	glVertex3f((GLfloat)x, (GLfloat)y, getHeight(x, y));
}

/* Draw the landscape */
void drawLandscape()
{
	int x, y;
	int p = (1 << detail);

	if (p > NSIDE - 1) p = NSIDE - 1;
	if (p < 1) p = 1;

	for (y = 0; y < (NSIDE - p); y += p) 
	{
		glBegin(GL_TRIANGLE_STRIP);
		placeGroundVertex(0, y, p);

		for (x = 0; x < (NSIDE - p); x += p) 
		{
			placeGroundVertex(x, y + p, p);
			placeGroundVertex(x + p, y, p);
		}

		placeGroundVertex(NSIDE - 1, y + p, p);
		glEnd();
	}
}




void setViewPoint()
{
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	glTranslatef(0,-height,-distance);
	glRotatef(-90+alt,1,0,0);
	glRotatef(azim,0,0,1);
}

void setupLighting ( float x, float y, float z )
{ 
	GLfloat
		lightdir[4] = { x, y, z, 0 },
		ambient[] = { 0.1, 0.1, 0.1, 0.1 },
		diffuse[] = { 1.0, 1.0, 1.0, 1.0 };

	/* note that the direction is a VECTOR, not a position, so the last 
	element is 0 */

	/* setting up incident, diffuse and ambient parameters of lighting model */

	glLightfv(GL_LIGHT0, GL_POSITION, lightdir);
	glLightfv(GL_LIGHT0, GL_AMBIENT, ambient);
	glLightfv(GL_LIGHT0, GL_DIFFUSE, diffuse);
	//glLightfv(GL_LIGHT0, GL_SPECULAR, diffuse);

	/* enable shading */
	glEnable(GL_SMOOTH);
	/* enable lighting */
	glEnable(GL_LIGHTING);
	/* enable light 0 */
	glEnable(GL_LIGHT0);
}


/* initLandscape */
void initLandscape(GLfloat A[NSIDE][NSIDE])
{
	int x, y;

	for (y = 0; y < NSIDE; y++) {
		for (x = 0; x < NSIDE; x++) {
			A[x][y] = 0.0;
		}
	}
}


double S(double delta)
{
	static double alpha = 0.5;
	static double c = 0.3;

	return c * ((pow(delta, -(log(alpha)) / log(2))));
}


/* Elevate the landscape */
void elevateLandscape(GLfloat A[NSIDE][NSIDE], int s)
{
	int i, x, y;

	// Do at many as s randomizing iterations
	for (i = 1; i <= s; i++) {
		// Calculate the step size of this iteration
		// (by definition, NSIDE-1 is always devidable by 2^(i-1))
		int p = (NSIDE - 1) / (1 << (i - 1));

		// Randomize field
		for (y = 0; y < NSIDE; y += p) {
			for (x = 0; x < NSIDE; x += p) {
				if (x < NSIDE - p) {
					//float rnd = (float)(rand()%100) / 100 - 0.5;
					float rnd = gasdev(&idum);
					A[x + p/2][y] = (A[x][y] + A[x + p][y]) / 2 + S(p) * rnd;
				}
				if (y < NSIDE - p) {
					//float rnd = (float)(rand()%100) / 100 - 0.5;
					float rnd = gasdev(&idum);
					A[x][y + p/2] = (A[x][y] + A[x][y + p]) / 2 + S(p) * rnd;
				}
			}
		}

		// Take care of midpoints
		for (y = 0; y < NSIDE - p; y += p) {
			for (x = 0; x < NSIDE - p; x += p) {
				//float rnd = (float)(rand()%100) / 100 - 0.5;
				float rnd = gasdev(&idum);
				A[x + p/2][y + p/2] = 
					(A[x + p/2][y] +
					A[x][y + p/2] +
					A[x + p][y + p/2] +
					A[x + p/2][y + p]) / 4 + 
					S(p) * rnd;
			}
		}
	}

	// Calculate minHeight and maxHeight
	minHeight = A[0][0];
	maxHeight = A[0][0];
	for (y = 0; y < NSIDE; y++) {
		for (x = 0; x < NSIDE; x++) {
			if (A[x][y] < minHeight) minHeight = A[x][y];
			if (A[x][y] > maxHeight) maxHeight = A[x][y];
		}
	}

	// Calculate water height
	waterHeight = ((2*minHeight + maxHeight) / 3);
}


void display (void)
{
	/* Variables for landscape drawing */
	GLfloat fogColor[4] = { 0.5, 0.5, 0.5, 1.0 };

	/* clear display and depth buffer     */ 
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	/* set up correct view point and lighting */
	setViewPoint();                
	setupLighting(-0.5,-1.0,1.0);

	/* Enable automatic normal scaling */
	glEnable(GL_NORMALIZE);

	/* draw scene */

	// Scale the landscape to the right proportions
	glPushMatrix();                                         // Save current matrix transformations
	glScalef(1.5 / NSIDE, 1.5 / NSIDE, 1.5 / NSIDE);        // Scale landscape down
	glTranslatef(-0.5 * (NSIDE-1), -0.5 * (NSIDE-1), 0.0);  // Translate landscape to center

	// Enable fog for additional realism
	glFogf(GL_FOG_DENSITY, 1.0);
	glFogf(GL_FOG_START, 1.5);
	glFogf(GL_FOG_END, 2.5);
	glFogf(GL_FOG_MODE, GL_LINEAR);
	glFogfv(GL_FOG_COLOR, fogColor);
	glEnable(GL_FOG);

	// Draw the landscape
	drawLandscape();

	// Draw the water plane
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glEnable(GL_BLEND);
	glBegin(GL_POLYGON);
	glNormal3f(0.0, 0.0, 1.0);
	glColor4f(0.0, 0.0, 1.0, 0.5);
	glVertex3f(0.0, 0.0, waterHeight);
	glVertex3f(0.0, NSIDE - 1, waterHeight);
	glVertex3f(NSIDE - 1, NSIDE - 1, waterHeight);
	glVertex3f(NSIDE - 1, 0.0, waterHeight);
	glEnd();
	glDisable(GL_BLEND);

	glDisable(GL_FOG);
	glPopMatrix();                                          // Restore matrix transformations

	/* swap front and back draw buffers */ 
	glutSwapBuffers();
}



void key(unsigned char k, int x, int y)
{
	switch (k) {
	case 'w' : alt_speed  += 1.0; break;
	case 's' : alt_speed  -= 1.0; break;
	case 'd' : azim_speed += 1.0; break;
	case 'a' : azim_speed -= 1.0; break;
	case '+' : detail--; if (detail < 0)     detail = 0;     break;
	case '-' : detail++; if (detail > STEPS) detail = STEPS; break;
	case 'r' :
	  initLandscape(ground);
	  elevateLandscape(ground, STEPS);
	  break;
	case 'q' : exit(0);
	  break;
	}
}

void mouse(int btn, int state, int x, int y)
{
	if (btn == GLUT_LEFT_BUTTON   && state == GLUT_DOWN) speed = -0.02;
	if (btn == GLUT_RIGHT_BUTTON  && state == GLUT_DOWN) speed = 0.02;
	if (btn == GLUT_MIDDLE_BUTTON && state == GLUT_DOWN) exit(0);
	if (state == GLUT_UP) speed = 0.0;
}

void idle()
{
	azim += azim_speed;
	distance += speed;
	alt += alt_speed;
	fase -= 0.6;

	glutPostRedisplay();
}

int main(int argc, char **argv)
{ 
	glutInit(&argc, argv);
	/* Initialize display mode with double buffering (animation)  
	and z buffer (hidden surface elimination)                */

	printf("Initializing landscape[%d][%d]\n", NSIDE, NSIDE);
	initLandscape(ground);
	elevateLandscape(ground, STEPS);

	glutInitDisplayMode(GLUT_DOUBLE | GLUT_RGB);

	glutInitWindowSize(winwidth, winheight);
	glutCreateWindow("Fracland: the magic wonderland by Laurens and Bjorn");

	glEnable(GL_DEPTH_TEST); /* Enable hidden surface removal */

	glEnable(GL_COLOR_MATERIAL);

	glClearColor(0.0, 0.0, 0.0, 0.0); /* black background */

	glMatrixMode(GL_PROJECTION);

	glLoadIdentity();
	glFrustum(-0.1,0.1,-0.1,0.1,0.1,60);

	glMatrixMode(GL_MODELVIEW);

	glutDisplayFunc(display);
	glutIdleFunc(idle);
	glutKeyboardFunc(key);
	glutMouseFunc(mouse);

	glutMainLoop();

	return 0;
}
